<?php
/**
 * Library Source class
 *
 * @package InventSlider
 * @version 1.0.8
 */
namespace DynamicLayers\InventSlider\Classes\Library;

use Elementor\TemplateLibrary\Source_Base;
use DynamicLayers\InventSlider\Classes\Helper;

defined( 'ABSPATH' ) || die();

class Source extends Source_Base {

	/**
	 * Template library transient key
	 */
	const LIBRARY_CACHE_KEY = 'invent_library_cache';

	/**
	 * Template info API URL
	 */
	const TEMPLATES_INFO_API_URL = 'https://addons.dynamiclayers.net/wp-json/rh/v1/invent-slider/templates-info';

	/**
	 * Template data API URL
	 */
	const TEMPLATE_DATA_API_URL = 'https://addons.dynamiclayers.net/invent-slider/wp-json/dl/v1/templates/';

	public function get_id() {
		return 'invent-slider-library';
	}

	public function get_title() {
		return __( 'Invent Slider Library', 'invent-slider' );
	}

	public function register_data() {}

	public function save_item( $template_data ) {
		return new \WP_Error( 'invalid_request', 'Cannot save template to a invent slider library' );
	}

	public function update_item( $new_data ) {
		return new \WP_Error( 'invalid_request', 'Cannot update template to a invent slider library' );
	}

	public function delete_template( $template_id ) {
		return new \WP_Error( 'invalid_request', 'Cannot delete template from a invent slider library' );
	}

	public function export_template( $template_id ) {
		return new \WP_Error( 'invalid_request', 'Cannot export template from a invent slider library' );
	}

	/**
	 * Get remote templates
	 * @param array $args
	 * @return array
	 */
	public function get_items( $args = [] ) {
		$library_data = self::get_library_data();

		$templates = [];

		if ( ! empty( $library_data['templates'] ) ) {
			foreach ( $library_data['templates'] as $template_data ) {
				$templates[] = $this->prepare_template( $template_data );
			}
		}

		return $templates;
	}

	/**
	 * Get tags
	 * @return array
	 */
	public function get_tags() {
		$library_data = self::get_library_data();
		return ! empty( $library_data['tags'] ) ? $library_data['tags'] : [];
	}

	/**
	 * Get type tags
	 * @return array
	 */
	public function get_type_tags() {
		$library_data = self::get_library_data();
		return ! empty( $library_data['type_tags'] ) ? $library_data['type_tags'] : [];
	}

	/**
	 * Prepare template items to match model
	 * @param array $template_data
	 * @return array
	 */
	private function prepare_template( array $template_data ) {
		return [
			'template_id' => $template_data['id'],
			'title'       => $template_data['title'],
			'type'        => $template_data['type'],
			'thumbnail'   => $template_data['thumbnail'],
			'date'        => $template_data['created_at'],
			'tags'        => $template_data['tags'],
			'isPro'       => $template_data['is_pro'],
			'url'         => $template_data['url'],
		];
	}

	/**
	 * Fetch and cache library data using transients
	 * @param bool $force_update
	 * @return array|false
	 */
	private static function request_library_data( $force_update = false ) {
		$transient_key = self::LIBRARY_CACHE_KEY;
		$data = get_transient( $transient_key );

		if ( $force_update || false === $data ) {
			$timeout  = $force_update ? 25 : 8;
			$response = wp_remote_get( self::TEMPLATES_INFO_API_URL, [ 'timeout' => $timeout ] );

			if ( is_wp_error( $response ) || 200 !== (int) wp_remote_retrieve_response_code( $response ) ) {
				delete_transient( $transient_key );
				return false;
			}

			$data = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( empty( $data ) || ! is_array( $data ) ) {
				delete_transient( $transient_key );
				return false;
			}

			// Cache for 10 days
			set_transient( $transient_key, $data, 10 * DAY_IN_SECONDS );
		}

		return $data;
	}

	/**
	 * Retrieve cached or refreshed library data
	 * @param bool $force_update
	 * @return array
	 */
	public static function get_library_data( $force_update = false ) {
		$data = self::request_library_data( $force_update );
		return ! empty( $data ) ? $data : [];
	}

	/**
	 * Get remote template
	 * @param int $template_id
	 * @return array
	 */
	public function get_item( $template_id ) {
		$templates = $this->get_items();
		return $templates[ $template_id ] ?? [];
	}

	/**
	 * Request template data from remote server
	 * @param int $template_id
	 * @return string|false
	 */
	public static function request_template_data( $template_id ) {
		if ( empty( $template_id ) ) {
			return;
		}

		$body = [
			'home_url' => trailingslashit( home_url() ),
			'version'  => INVENT_SLIDER_VERSION,
		];

		$response = wp_remote_get(
			self::TEMPLATE_DATA_API_URL . $template_id,
			[
				'body'    => $body,
				'timeout' => 25,
			]
		);

		return wp_remote_retrieve_body( $response );
	}

	/**
	 * Retrieve and process template data
	 * @param array $args
	 * @param string $context
	 * @return array
	 */
	public function get_data( array $args, $context = 'display' ) {
		$data = self::request_template_data( $args['template_id'] );
		$data = json_decode( $data, true );

		if ( empty( $data ) || empty( $data['content'] ) ) {
			throw new \Exception( __( 'Template does not have any content', 'invent-slider' ) );
		}

		$data['content'] = $this->replace_elements_ids( $data['content'] );
		$data['content'] = $this->process_export_import_content( $data['content'], 'on_import' );

		$post_id  = $args['editor_post_id'];
		$document = Helper::elementor_instance()->documents->get( $post_id );

		if ( $document ) {
			$data['content'] = $document->get_elements_raw_data( $data['content'], true );
		}

		return $data;
	}
}
